classdef DynamicElement < matlab.mixin.Copyable & matlab.mixin.SetGetExactNames  % SetGetExactNames is not supported in MATLAB2015
% DYNAMICELEMENT An element that evolves dynamically over time
%
% Properties:
%   label       The name of the element (string), e.g., 'x.state1'
% 	def 		A definition of the element, for example an ODE (string)
% 	val 		The dynamic values of the array (double).
%               This can be in one of two forms:
%                   If a scalar value is used, it is a static value
%                       (parameter, initial value for a state)
%                   Otherwise, a two-column matrix is expected, where the
%                   first column represents time, and the second column
%                   represents values throughout this time

% David Katzin, Wageningen University
% david.katzin@wur.nl
    
    properties 
        label
        def
        val
        func
    end

    methods
        function obj = DynamicElement(varargin)
        % DYNAMICELEMENT constructor for DynamicElement
        % Usage:
        %   DynamicElement()
        %       Creates an empty DynamicElement
        %   DynamicElement(de)
        %       When de is a DynamicElement, creates a copy of the DynamicElement de
        %   DynamicElement(lab)
        %       When lab is a string, creates a Dynamic with label and def
        %       as the given argument
        %   DynamicElement(lab, val)
        %       Sets label and def as the given lab, val as the given val
        %   DynamicElement(label, val, def) 
        %       Creates a DynamicElement with the given properties.
        %   DynamicElement(label, val, def, func) 
        %       Creates a DynamicElement with the given properties.
            
            if nargin ==  1 
                if isa(varargin{1}, 'DynamicElement')
                    deIn = varargin{1};
                    obj.def = deIn.def;
                    obj.val = deIn.val;
                    obj.func = deIn.func;
                    obj.label = deIn.label;
                else
                    obj.def = varargin{1};
                    obj.label = varargin{1};
                    if isempty(varargin{1})
                        obj.func = [];
                    else
                        obj.func = str2func(['@(x,a,u,d,p)' varargin{1}]);
                    end
                end
            elseif nargin > 1 && ~isa(varargin{1}, 'DynamicElement')
                obj.def = varargin{1};
                obj.val = varargin{2};
                obj.label = varargin{1};
                if nargin > 2
                    obj.def = varargin{3};
                end
             % define the function field
                if nargin == 2
                    try
                        obj.func = str2func(['@(x,a,u,d,p)' varargin{1}]);
                    catch
                        error('Could not define function %s', varargin{1});
                    end
                elseif nargin == 3
                    try
                        obj.func = str2func(['@(x,a,u,d,p)' varargin{3}]);
                    catch
                        error('Could not define function %s', varargin{3});
                    end
                else
                    obj.func = varargin{4};
                end                
            end
        end
        
        %% set methods 
        % These are called automatically when using the usual syntax to set
        % variables. Actually not necessary to define but are here so 
        % that validation can be performed if needed
		% (currently no validation is performed)
        
        function set.def(obj,def)
            obj.def = def;
        end
        
        function set.val(obj,val)
            obj.val = val;
        end
		
        %% plot 
        function plot(obj, varargin)
           s = size(obj.val);
           if s(2) == 2
              plot(obj.val(:,1),obj.val(:,2),varargin{:}); 
           else % nothing to plot
              plot(0);
           end
        end
        
        %% scatter 
        function scatter(obj)
           s = size(obj.val);
           if s(2) == 2
              scatter(obj.val(:,1),obj.val(:,2)); 
           end
        end
        
        %% trapz
        function sum = trapz(obj)
        % integral of the values of obj
            sum = trapz(obj.val(:,1), obj.val(:,2));
        end
        
        %% rmse
        function r = rmse(obj1, obj2)
        % RMSE between two objects
            if ~isequal(obj1.val(:,1), obj2.val(:,1))
                warning('Timelines of objects not equal');
                r = [];
            else
                r = sqrt(mean((obj1.val(:,2)-obj2.val(:,2)).^2));
            end
        end
        
        %% rrmse
        function r = rrmse(obj1, obj2)
        % relative RMSE between two objects [%] (relative to the first
        % object)
            if ~isequal(obj1.val(:,1), obj2.val(:,1))
                warning('Timelines of objects not equal');
                r = [];
            else
                r = 100*rmse(obj1,obj2)/mean(obj1.val(:,2));
            end
        end
        
		%% Arithmetic operations
		% Methods for arithmetic operations between DynamicElements      
        function de = lt(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '<');
        end
        function de = gt(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '>');
        end
        function de = le(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '<=');
        end
        function de = ge(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '>=');
        end
        function de = ne(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '~=');
        end
        function de = eq(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '==');
        end
        function de = and(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '&');
        end
        function de = or(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '|');
        end
        function de = plus(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '+');
        end
        function de = minus(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '-');
        end
        function de = times(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '.*');
        end
		function de = mtimes(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '*');
        end
        function de = rdivide(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, './');
        end
		function de = mrdivide(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '/');
        end
        function de = power(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '.^');
        end
		function de = mpower(obj1, obj2)
            de = binaryArithmetic(obj1, obj2, '^');
        end
        function de = max(obj1, obj2)
            de = binaryFunction(obj1, obj2, 'max');
        end
        function de = min(obj1, obj2)
            de = binaryFunction(obj1, obj2, 'min');
        end
        function de = not(obj)
			de = unaryFunction(obj,'~');
        end
		function de = abs(obj)
			de = unaryFunction(obj,'abs');
        end
        function de = exp(obj)
			de = unaryFunction(obj,'exp');
        end
        function de = sqrt(obj)
			de = unaryFunction(obj,'sqrt');
        end
        function de = floor(obj)
            de = unaryFunction(obj,'floor');
        end
		function de = ceil(obj)
            de = unaryFunction(obj,'ceil');
        end
        function de = sign(obj)
            de = unaryFunction(obj,'sign');
        end
        function de = uminus(obj)
			de = DynamicElement(['-' obj.def ''], -obj.val);
        end
        function de = cos(obj)
            de = unaryFunction(obj,'cos');
        end
        function de = sin(obj)
            de = unaryFunction(obj,'sin');
        end
		function de = cosd(obj)
            de = unaryFunction(obj,'cosd');
        end
        function de = sind(obj)
            de = unaryFunction(obj,'sind');
        end
        function de = cumsum(obj)
            de = unaryFunction(obj,'cumsum');
        end
        function de = nthroot(obj, num)
            de = binaryFunction(obj, num, 'nthroot');
        end
        function de = smooth(obj, num)
            de = binaryFunction(obj, num, 'smooth');
        end
        function de = mod(obj, num)
            de = binaryFunction(obj, num, 'mod');
        end
        function de = log(obj)
            de = unaryFunction(obj, 'log');
        end
        function avg = mean(obj)
            if isscalar(obj.val)
                avg = obj.val;
            else
                avg = mean(obj.val(:,2));
            end
        end
        
        % divide without brackets
		function de = divNoBracks(obj1, obj2)
			de = binaryArithmetic(obj1, obj2, './', false);
		end
		
		% multiply without brackets
		function de = mulNoBracks(obj1, obj2)
			de = binaryArithmetic(obj1, obj2, '.*', false);
		end

        
        function de = binaryArithmetic(obj1, obj2, operator, bracks)
		% Create a new dynamic element of the form <obj1> <operator> <obj2>
        % the definition will be (<def1>) <operator> (<def2>)
        % if bracks==false, the definition will be <def1> <operator> <def2>
        % (risky to do, but helps shorten new defs)
        % 
        % at least one of the two objects must be a DynamicElement. One of
        % them may be a scalar number.
        %
        % the val will be a result of the binary operation:
        %   if both DynamicElements have a scalar val, a new scalar val
        %       will be calculated
        %   if both DynamicElements have a time trajectory val, with the same
        %       time values, a new trajectory will be (elementwise) calculated
        %   if one val is scalar and one val is a trajectory, a new
        %       trajectory will be calculated
        
			if ~exist('bracks', 'var')
				bracks = true; % default is use brackets
			end
			
			[def1, def2, val1, val2, lab1, lab2] = getProperties(obj1,obj2);
			
            % add brackets
            if isa(obj1, 'DynamicElement') && ~strcmp(operator, '+') && ~strcmp(operator, '-') && bracks
                def1 = ['(' def1 ')'];
                lab1 = ['(' lab1 ')'];
            end
            if isa(obj2, 'DynamicElement') && ~strcmp(operator, '+') && bracks
                def2 = ['(' def2 ')'];
                lab2 = ['(' lab2 ')'];
            end
            
            % set definition
            definition = [def1 operator def2];
            lab = [lab1 operator lab2];
			
            % set value
            size1 = size(val1);
            size2 = size(val2);
            
            if size1(1)==0 || size2(1)==0
            % empty values
            	value = [];
            elseif isequal(size1,[1 1]) && isequal(size2,[1 1])
            % scalar artihmetic    
            	eval(['value = val1' operator 'val2;']);
            elseif size1(2) == 2 && isequal(size2,[1 1])
            % 2-column matrix and scalar
                eval(['value = [val1(:,1) val1(:,2)' operator 'val2];']);
            elseif isequal(size1,[1 1]) && size2(2) == 2
            % scalar and 2-column matrix
                eval(['value = [val2(:,1) val1' operator 'val2(:,2)];']);
            elseif size1(2)==2 && size2(2) == 2
            % two 2-column matrices
                if ~isequal(val1(:,1),val2(:,1))
                    value = [];
                    warning('Timelines of values not equal, assigned empty value');
                else
                    eval(['value = [val1(:,1) val1(:,2)' operator 'val2(:,2)];']);
                end
            else
                value = [];
                warning('Could not perform arithmetic operation, assigned empty value');
            end
			
			de = DynamicElement(lab, value, definition);
        end
        
        function de = binaryFunction(obj1, obj2, func)
        % Create a new dynamic element of the form func(<obj1>, <obj2>)
        % Besides the format, works the same as binaryArithmetic
			
			% get definitions and values based on dataype of obj1, obj2
			[def1, def2, val1, val2, lab1, lab2] = getProperties(obj1,obj2);
            
            % set definition
            definition = [func '(' def1 ',' def2 ')'];
			lab = [func '(' lab1 ',' lab2 ')'];
            
            % set value
            size1 = size(val1);
            size2 = size(val2);
            
            if size1(1)==0 || size2(1)==0
            % empty values
            	value = [];
            elseif isequal(size1,[1 1]) && isequal(size2,[1 1])
            % scalar artihmetic    
                eval(['value = ' func '(val1,val2);']);
            elseif size1(2) == 2 && isequal(size2,[1 1])
            % 2-column matrix and scalar
                eval(['value = [val1(:,1) ' func '(val1(:,2),val2)];']);
            elseif isequal(size1,[1 1]) && size2(2) == 2
            % scalar and 2-column matrix
                eval(['value = [val2(:,1) ' func '(val1,val2(:,2))];']);
            elseif size1(2)==2 && size2(2) == 2
            % two 2-column matrices
                if ~isequal(val1(:,1),val2(:,1))
                    value = [];
                    warning('Timelines of values not equal, assigned empty value');
                else
                    eval(['value = [val1(:,1) ' func '(val1(:,2),val2(:,2))];']);
                end
            else
                value = [];
                warning('Could not perform binary function, assigned empty value');
            end
			
			de = DynamicElement(lab, value, definition);
        end
        
        function de = unaryFunction(obj, func)
        % Create a new dynamic element of the form func(<obj>)
        % Besides this, works the same as binaryFunction
        
            definition = [func '(' obj.def ')'];
            lab = [func '(' obj.label ')'];
            
            % set value
            valSize = size(obj.val);
            if valSize(1)==0
            % empty value
            	value = [];
            elseif isequal(valSize,[1 1])
            % scalar 
                value = eval([func '(obj.val);']);
            elseif valSize(2) == 2
            % 2-column matrix 
                eval(['value = [obj.val(:,1) ' func '(obj.val(:,2))];']);
            else
                value = [];
                warning('Could not perform unary function, assigned empty value');
            end
            
            de = DynamicElement(lab, value, definition);
        end
        
        function [def1, def2, val1, val2, lab1, lab2] = getProperties(obj1,obj2)
        % Get the properties of obj1 and obj2
        % If the object is a DynamicElement simply get the corresponding
        % fields
        % If the object is numeric the val will be the value, and the def
        % will be a string representing that value
        
            if isa(obj1, 'DynamicElement') && isa(obj2, 'DynamicElement')
                def1 = obj1.def;
				val1 = obj1.val;
				def2 = obj2.def;
				val2 = obj2.val;
                lab1 = obj1.label;
                lab2 = obj2.label;
			elseif isnumeric(obj1)
				def1 = num2str(obj1);
				val1 = obj1;
                lab1 = num2str(obj1);
				def2 = obj2.def;
				val2 = obj2.val;
                lab2 = obj2.label;
			elseif isnumeric(obj2)
                def1 = obj1.def;
				val1 = obj1.val;
                lab1 = obj1.label;
				def2 = num2str(obj2);
				val2 = obj2;
                lab2 = num2str(obj2);
            else
                error('Wrong datatype for binary operation');
            end
        end
    end
end

